#include <stdio.h>
#include <stdlib.h>

#ifdef USE_NLLIBC
#include <nllibc.h>
#endif

#include "config.h"
#include "lib.h"
#include "type.h"
#include "asm_symbol.h"
#include "asm_code.h"
#include "asm_code_powerpc.h"

static char *load_by_size(model_t model)
{
  if (model != NULL) {
    switch (model_get_size(model)) {
    case -1: ERREXIT(1);
    case  1: return "lbz";
    case  2: return "lhz";
    default: break;
    }
  }
  return "lwz";
}

static char *store_by_size(model_t model)
{
  if (model != NULL) {
    switch (model_get_size(model)) {
    case -1: ERREXIT(1);
    case  1: return "stb";
    case  2: return "sth";
    default: break;
    }
  }
  return "stw";
}

static void get_value(FILE *out, long value)
{
  ASM_CODE_OUT(out, "\tlis\t%%r3, 0x%lx\n", (value >> 16) & 0xffff);
  ASM_CODE_OUT(out, "\tori\t%%r3, %%r3, 0x%lx\n", value & 0xffff);
}

static void get_value_r1(FILE *out, long value)
{
  ASM_CODE_OUT(out, "\tlis\t%%r4, 0x%lx\n", (value >> 16) & 0xffff);
  ASM_CODE_OUT(out, "\tori\t%%r4, %%r4, 0x%lx\n", value & 0xffff);
}

static void get_address_stack(FILE *out, int offset)
{
  ASM_CODE_OUT(out, "\taddi\t%%r3, %%r1, 0x%x\n", offset);
}

static int get_address_stack_r1(FILE *out, int offset)
{
  ASM_CODE_OUT(out, "\taddi\t%%r4, %%r1, 0x%x\n", offset);
  return 0;
}

static void get_address(FILE *out, char *label)
{
  ASM_CODE_OUT(out, "\tlis\t%%r3, %s@ha\n", label);
  ASM_CODE_OUT(out, "\taddi\t%%r3, %%r3, %s@l\n", label);
}

static void add_address(FILE *out, int offset)
{
  ASM_CODE_OUT(out, "\taddi\t%%r3, %%r3, 0x%x\n", offset);
}

static void get_r1(FILE *out)
{
  ASM_CODE_OUT(out, "\tmr\t%%r3, %%r4\n");
}

static void set_r1(FILE *out)
{
  ASM_CODE_OUT(out, "\tmr\t%%r4, %%r3\n");
}

static void memory_load(FILE *out, int offset, model_t model)
{
  ASM_CODE_OUT(out, "\t%s\t%%r3, 0x%x(%%r4)\n", load_by_size(model), offset);
}

static void memory_store(FILE *out, int offset, model_t model)
{
  ASM_CODE_OUT(out, "\t%s\t%%r3, 0x%x(%%r4)\n", store_by_size(model), offset);
}

static void stack_load(FILE *out, int offset, model_t model)
{
  ASM_CODE_OUT(out, "\t%s\t%%r3, 0x%x(%%r1)\n", load_by_size(model), offset);
}

static void stack_store(FILE *out, int offset, model_t model)
{
  ASM_CODE_OUT(out, "\t%s\t%%r3, 0x%x(%%r1)\n", store_by_size(model), offset);
}

static void stack_load_r1(FILE *out, int offset, model_t model)
{
  ASM_CODE_OUT(out, "\t%s\t%%r4, 0x%x(%%r1)\n", load_by_size(model), offset);
}

static void stack_store_r1(FILE *out, int offset, model_t model)
{
  ASM_CODE_OUT(out, "\t%s\t%%r4, 0x%x(%%r1)\n", store_by_size(model), offset);
}

static void stack_expand(FILE *out, int size)
{
  ASM_CODE_OUT(out, "\taddi\t%%r1, %%r1, -0x%x\n", size);
}

static void stack_reduce(FILE *out, int size)
{
  ASM_CODE_OUT(out, "\taddi\t%%r1, %%r1, 0x%x\n", size);
}

static char *funccall_regs[] = {
  "r3", "r4", "r5", "r6", "r7", "r8", "r9", "r10"
};

static void funccall_reg_load(FILE *out, int number)
{
  ASM_CODE_OUT(out, "\tmr\t%%r3, %%%s\n", funccall_regs[number]);
}

static void funccall_reg_store(FILE *out, int number)
{
  ASM_CODE_OUT(out, "\tmr\t%%%s, %%r3\n", funccall_regs[number]);
}

static char *tmp_regs[] = {
  "r11", "r12",
};

static void tmp_reg_load(FILE *out, int number)
{
  ASM_CODE_OUT(out, "\tmr\t%%r3, %%%s\n", tmp_regs[number]);
}

static void tmp_reg_save(FILE *out, int number)
{
  ASM_CODE_OUT(out, "\tmr\t%%%s, %%r3\n", tmp_regs[number]);
}

static void tmp_reg_load_r1(FILE *out, int number)
{
  ASM_CODE_OUT(out, "\tmr\t%%r4, %%%s\n", tmp_regs[number]);
}

static void tmp_reg_save_r1(FILE *out, int number)
{
  ASM_CODE_OUT(out, "\tmr\t%%%s, %%r4\n", tmp_regs[number]);
}

static int sign_extension_char(FILE *out)
{
  ASM_CODE_OUT(out, "\textsb\t%%r3, %%r3\n");
  return 0;
}

static int sign_extension_uchar(FILE *out)
{
  ASM_CODE_OUT(out, "\tclrlwi\t%%r3, %%r3, 24\n");
  return 0;
}

static int sign_extension_short(FILE *out)
{
  ASM_CODE_OUT(out, "\textsh\t%%r3, %%r3\n");
  return 0;
}

static int sign_extension_ushort(FILE *out)
{
  ASM_CODE_OUT(out, "\tclrlwi\t%%r3, %%r3, 16\n");
  return 0;
}

static int sign_extension_int(FILE *out)
{
  return -1;
}

static int sign_extension_uint(FILE *out)
{
  return -1;
}

static int calc_inv(FILE *out, model_t model, model_t model_arg0)
{
  ASM_CODE_OUT(out, "\tnot\t%%r3, %%r3\n");
  return 0;
}

static int calc_minus(FILE *out, model_t model, model_t model_arg0)
{
  ASM_CODE_OUT(out, "\tneg\t%%r3, %%r3\n");
  return 0;
}

static int calc_op1(FILE *out, c_type_t type, model_t model, model_t model_arg0)
{
  return -1;
}

static void calc_add(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\tadd\t%%r3, %%r3, %%r4\n");
}

static void calc_sub(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\tsubf\t%%r3, %%r4, %%r3\n");
}

static void calc_and(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\tand\t%%r3, %%r3, %%r4\n");
}

static void calc_or(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\tor\t%%r3, %%r3, %%r4\n");
}

static int calc_xor(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\txor\t%%r3, %%r3, %%r4\n");
  return 0;
}

static int calc_mul(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\tmullw\t%%r3, %%r3, %%r4\n");
  return 0;
}

static int calc_div(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  if (model_arg0->flags & MODEL_FLAG_UNSIGNED) {
    ASM_CODE_OUT(out, "\tdivwu\t%%r3, %%r3, %%r4\n");
  } else {
    ASM_CODE_OUT(out, "\tdivw\t%%r3, %%r3, %%r4\n");
  }
  return 0;
}

static int calc_mod(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  if (model_arg0->flags & MODEL_FLAG_UNSIGNED) {
    ASM_CODE_OUT(out, "\tdivwu\t%%r0, %%r3, %%r4\n");
  } else {
    ASM_CODE_OUT(out, "\tdivw\t%%r0, %%r3, %%r4\n");
  }
  ASM_CODE_OUT(out, "\tmullw\t%%r0, %%r0, %%r4\n");
  ASM_CODE_OUT(out, "\tsubf\t%%r3, %%r0, %%r3\n");
  return 0;
}

static int calc_llshift(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\tslw\t%%r3, %%r3, %%r4\n");
  return 0;
}

static int calc_rashift(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\tsraw\t%%r3, %%r3, %%r4\n");
  return 0;
}

static int calc_rlshift(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\tsrw\t%%r3, %%r3, %%r4\n");
  return 0;
}

static int calc_op2(FILE *out, c_type_t type, model_t model, model_t model_arg0, model_t model_arg1)
{
  return -1;
}

static void branch(FILE *out, char *label)
{
  ASM_CODE_OUT(out, "\tb\t%s\n", label);
}

static int branch_zero(FILE *out, char *label)
{
  ASM_CODE_OUT(out, "\tcmpwi\t%%r3, 0\n");
  ASM_CODE_OUT(out, "\tbeq\t%s\n", label);
  return 0;
}

static int branch_nzero(FILE *out, char *label)
{
  ASM_CODE_OUT(out, "\tcmpwi\t%%r3, 0\n");
  ASM_CODE_OUT(out, "\tbne\t%s\n", label);
  return 0;
}

static int branch_cmp_eq(FILE *out, char *label, model_t model)
{
  ASM_CODE_OUT(out, "\tcmpw\t%%r3, %%r4\n");
  ASM_CODE_OUT(out, "\tbeq\t%s\n", label);
  return 0;
}

static int branch_cmp_ne(FILE *out, char *label, model_t model)
{
  ASM_CODE_OUT(out, "\tcmpw\t%%r3, %%r4\n");
  ASM_CODE_OUT(out, "\tbne\t%s\n", label);
  return 0;
}

static int branch_cmp_lt(FILE *out, char *label, model_t model)
{
  ASM_CODE_OUT(out, "\tcmpw\t%%r3, %%r4\n");
  ASM_CODE_OUT(out, "\tblt\t%s\n", label);
  return 0;
}

static int branch_cmp_gt(FILE *out, char *label, model_t model)
{
  ASM_CODE_OUT(out, "\tcmpw\t%%r3, %%r4\n");
  ASM_CODE_OUT(out, "\tbgt\t%s\n", label);
  return 0;
}

static int branch_cmp_le(FILE *out, char *label, model_t model)
{
  ASM_CODE_OUT(out, "\tcmpw\t%%r3, %%r4\n");
  ASM_CODE_OUT(out, "\tble\t%s\n", label);
  return 0;
}

static int branch_cmp_ge(FILE *out, char *label, model_t model)
{
  ASM_CODE_OUT(out, "\tcmpw\t%%r3, %%r4\n");
  ASM_CODE_OUT(out, "\tbge\t%s\n", label);
  return 0;
}

static int branch_cmp_ult(FILE *out, char *label, model_t model)
{
  ASM_CODE_OUT(out, "\tcmplw\t%%r3, %%r4\n");
  ASM_CODE_OUT(out, "\tblt\t%s\n", label);
  return 0;
}

static int branch_cmp_ugt(FILE *out, char *label, model_t model)
{
  ASM_CODE_OUT(out, "\tcmplw\t%%r3, %%r4\n");
  ASM_CODE_OUT(out, "\tbgt\t%s\n", label);
  return 0;
}

static int branch_cmp_ule(FILE *out, char *label, model_t model)
{
  ASM_CODE_OUT(out, "\tcmplw\t%%r3, %%r4\n");
  ASM_CODE_OUT(out, "\tble\t%s\n", label);
  return 0;
}

static int branch_cmp_uge(FILE *out, char *label, model_t model)
{
  ASM_CODE_OUT(out, "\tcmplw\t%%r3, %%r4\n");
  ASM_CODE_OUT(out, "\tbge\t%s\n", label);
  return 0;
}

static int branch_cmp(FILE *out, char *label, c_type_t type, model_t model)
{
  return -1;
}

static int function_call(FILE *out, char *label)
{
  ASM_CODE_OUT(out, "\tbl\t%s\n", label);
  return 0;
}

static int function_call_set(FILE *out)
{
  ASM_CODE_OUT(out, "\tmr\t%%r0, %%r3\n");
  return 0;
}

static int function_call_pointer(FILE *out)
{
  ASM_CODE_OUT(out, "\tmtlr\t%%r0\n");
  ASM_CODE_OUT(out, "\tblrl\n");
  return 0;
}

static void function_start(FILE *out)
{
  ASM_CODE_OUT(out, "\tstwu\t%%r1, -8(%%r1)\n");
  ASM_CODE_OUT(out, "\tmflr\t%%r0\n");
  ASM_CODE_OUT(out, "\tstw\t%%r0, 4(%%r1)\n");
}

static void function_end(FILE *out)
{
  ASM_CODE_OUT(out, "\tlwz\t%%r0, 4(%%r1)\n");
  ASM_CODE_OUT(out, "\tmtlr\t%%r0\n");
  ASM_CODE_OUT(out, "\taddi\t%%r1, %%r1, 8\n");
  ASM_CODE_OUT(out, "\tblr\n\n");
}

static void function_register_save(FILE *out)
{
  ASM_CODE_OUT(out, "\taddi\t%%r1, %%r1, -8\n");
  ASM_CODE_OUT(out, "\tstw\t%%r11, 0(%%r1)\n");
  ASM_CODE_OUT(out, "\tstw\t%%r12, 4(%%r1)\n");
}

static void function_register_load(FILE *out)
{
  ASM_CODE_OUT(out, "\tlwz\t%%r11, 0(%%r1)\n");
  ASM_CODE_OUT(out, "\tlwz\t%%r12, 4(%%r1)\n");
  ASM_CODE_OUT(out, "\taddi\t%%r1, %%r1, 8\n");
}

struct asm_code_callback asm_code_callback_powerpc = {
  NULL, /* type mark */

  4, /* word size */
  4, /* pointer size */
  8, /* funccall args reg number: r3/r4/r5/r6/r7/r8/r9/r10 (See funccall_regs[]) */
  VALUE_FUNCTION_ARGS_NUM, /* funccall args stack number */
  2, /* tmp reg number: r11/r12 (See tmp_regs[]) */
  VALUE_FUNCTION_ARGS_NUM, /* tmp stack number */
  2, /* function register number: r11/r12 (See function_register_save() */
  2, /* function saveparam number: SP chain and LR (See function_start() */
  16, /* stack align size */
  0, /* stack correct size */

  get_value,
  get_value_r1,
  get_address_stack,
  get_address_stack_r1,
  get_address,
  add_address,
  get_r1,
  set_r1,

  memory_load,
  memory_store,
  stack_load,
  stack_store,
  stack_load_r1,
  stack_store_r1,
  stack_expand,
  stack_reduce,

  funccall_reg_load,
  funccall_reg_store,
  tmp_reg_load,
  tmp_reg_save,
  tmp_reg_load_r1,
  tmp_reg_save_r1,

  sign_extension_char,
  sign_extension_uchar,
  sign_extension_short,
  sign_extension_ushort,
  sign_extension_int,
  sign_extension_uint,

  calc_inv,
  calc_minus,

  calc_op1,

  calc_add,
  calc_sub,
  calc_and,
  calc_or,
  calc_xor,

  calc_mul,
  calc_div,
  calc_mod,
  calc_llshift,
  calc_rashift,
  calc_rlshift,

  calc_op2,

  branch,
  branch_zero,
  branch_nzero,

  branch_cmp_eq,
  branch_cmp_ne,

  branch_cmp_lt,
  branch_cmp_gt,
  branch_cmp_le,
  branch_cmp_ge,
  branch_cmp_ult,
  branch_cmp_ugt,
  branch_cmp_ule,
  branch_cmp_uge,

  branch_cmp,

  function_call,
  function_call_set,
  function_call_pointer,

  function_start,
  function_end,
  function_register_save,
  function_register_load,
};
