#include <stdio.h>
#include <stdlib.h>

#ifdef USE_NLLIBC
#include <nllibc.h>
#endif

#include "config.h"
#include "lib.h"
#include "type.h"
#include "asm_symbol.h"
#include "asm_code.h"
#include "asm_code_rx.h"

static char *load_by_size(model_t model)
{
  if (model != NULL) {
    switch (model_get_size(model)) {
    case -1: ERREXIT(1);
    case  1: return "mov.b";
    case  2: return "mov.w";
    default: break;
    }
  }
  return "mov.l";
}

static char *store_by_size(model_t model)
{
  if (model != NULL) {
    switch (model_get_size(model)) {
    case -1: ERREXIT(1);
    case  1: return "mov.b";
    case  2: return "mov.w";
    default: break;
    }
  }
  return "mov.l";
}

static void get_value(FILE *out, long value)
{
  ASM_CODE_OUT(out, "\tmov.l\t#0x%lx, r1\n", value);
}

static void get_value_r1(FILE *out, long value)
{
  ASM_CODE_OUT(out, "\tmov.l\t#0x%lx, r2\n", value);
}

static void get_address_stack(FILE *out, int offset)
{
  ASM_CODE_OUT(out, "\tmov.l\tr0, r1\n");
  ASM_CODE_OUT(out, "\tadd\t#0x%x, r1\n", offset);
}

static int get_address_stack_r1(FILE *out, int offset)
{
  ASM_CODE_OUT(out, "\tmov.l\tr0, r2\n");
  ASM_CODE_OUT(out, "\tadd\t#0x%x, r2\n", offset);
  return 0;
}

static void get_address(FILE *out, char *label)
{
  ASM_CODE_OUT(out, "\tmov.l\t#%s, r1\n", label);
}

static void add_address(FILE *out, int offset)
{
  ASM_CODE_OUT(out, "\tadd\t#0x%x, r1\n", offset);
}

static void get_r1(FILE *out)
{
  ASM_CODE_OUT(out, "\tmov.l\tr2, r1\n");
}

static void set_r1(FILE *out)
{
  ASM_CODE_OUT(out, "\tmov.l\tr1, r2\n");
}

static void memory_load(FILE *out, int offset, model_t model)
{
  ASM_CODE_OUT(out, "\t%s\t0x%x[r2], r1\n", load_by_size(model), offset);
}

static void memory_store(FILE *out, int offset, model_t model)
{
  ASM_CODE_OUT(out, "\t%s\tr1, 0x%x[r2]\n", store_by_size(model), offset);
}

static void stack_load(FILE *out, int offset, model_t model)
{
  ASM_CODE_OUT(out, "\t%s\t0x%x[r0], r1\n", load_by_size(model), offset);
}

static void stack_store(FILE *out, int offset, model_t model)
{
  ASM_CODE_OUT(out, "\t%s\tr1, 0x%x[r0]\n", store_by_size(model), offset);
}

static void stack_load_r1(FILE *out, int offset, model_t model)
{
  ASM_CODE_OUT(out, "\t%s\t0x%x[r0], r2\n", load_by_size(model), offset);
}

static void stack_store_r1(FILE *out, int offset, model_t model)
{
  ASM_CODE_OUT(out, "\t%s\tr2, 0x%x[r0]\n", store_by_size(model), offset);
}

static void stack_expand(FILE *out, int size)
{
  ASM_CODE_OUT(out, "\tsub\t#0x%x, r0\n", size);
}

static void stack_reduce(FILE *out, int size)
{
  ASM_CODE_OUT(out, "\tadd\t#0x%x, r0\n", size);
}

static char *funccall_regs[] = {
  "r1", "r2", "r3", "r4"
};

static void funccall_reg_load(FILE *out, int number)
{
  ASM_CODE_OUT(out, "\tmov.l\t%s, r1\n", funccall_regs[number]);
}

static void funccall_reg_store(FILE *out, int number)
{
  ASM_CODE_OUT(out, "\tmov.l\tr1, %s\n", funccall_regs[number]);
}

static char *tmp_regs[] = {
  "r7", "r8", "r9", "r10", "r11", "r12", "r13",
};

static void tmp_reg_load(FILE *out, int number)
{
  ASM_CODE_OUT(out, "\tmov.l\t%s, r1\n", tmp_regs[number]);
}

static void tmp_reg_save(FILE *out, int number)
{
  ASM_CODE_OUT(out, "\tmov.l\tr1, %s\n", tmp_regs[number]);
}

static void tmp_reg_load_r1(FILE *out, int number)
{
  ASM_CODE_OUT(out, "\tmov.l\t%s, r2\n", tmp_regs[number]);
}

static void tmp_reg_save_r1(FILE *out, int number)
{
  ASM_CODE_OUT(out, "\tmov.l\tr2, %s\n", tmp_regs[number]);
}

static int sign_extension_char(FILE *out)
{
  ASM_CODE_OUT(out, "\tshll\t#24, r1\n");
  ASM_CODE_OUT(out, "\tshar\t#24, r1\n");
  return 0;
}

static int sign_extension_uchar(FILE *out)
{
  ASM_CODE_OUT(out, "\tshll\t#24, r1\n");
  ASM_CODE_OUT(out, "\tshlr\t#24, r1\n");
  return 0;
}

static int sign_extension_short(FILE *out)
{
  ASM_CODE_OUT(out, "\tshll\t#16, r1\n");
  ASM_CODE_OUT(out, "\tshar\t#16, r1\n");
  return 0;
}

static int sign_extension_ushort(FILE *out)
{
  ASM_CODE_OUT(out, "\tshll\t#16, r1\n");
  ASM_CODE_OUT(out, "\tshlr\t#16, r1\n");
  return 0;
}

static int sign_extension_int(FILE *out)
{
  return -1;
}

static int sign_extension_uint(FILE *out)
{
  return -1;
}

static int calc_inv(FILE *out, model_t model, model_t model_arg0)
{
  ASM_CODE_OUT(out, "\tnot\tr1\n");
  return 0;
}

static int calc_minus(FILE *out, model_t model, model_t model_arg0)
{
  ASM_CODE_OUT(out, "\tneg\tr1\n");
  return 0;
}

static int calc_op1(FILE *out, c_type_t type, model_t model, model_t model_arg0)
{
  return -1;
}

static void calc_add(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\tadd\tr2, r1\n");
}

static void calc_sub(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\tsub\tr2, r1\n");
}

static void calc_and(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\tand\tr2, r1\n");
}

static void calc_or(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\tor\tr2, r1\n");
}

static int calc_xor(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\txor\tr2, r1\n");
  return 0;
}

static int calc_mul(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\tmul\tr2, r1\n");
  return 0;
}

static int calc_div(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  if (model_arg0->flags & MODEL_FLAG_UNSIGNED) {
    ASM_CODE_OUT(out, "\tdivu\tr2, r1\n");
  } else {
    ASM_CODE_OUT(out, "\tdiv\tr2, r1\n");
  }
  return 0;
}

static int calc_mod(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\tmov.l\tr1, r14\n");
  if (model_arg0->flags & MODEL_FLAG_UNSIGNED) {
    ASM_CODE_OUT(out, "\tdivu\tr2, r14\n");
  } else {
    ASM_CODE_OUT(out, "\tdiv\tr2, r14\n");
  }
  ASM_CODE_OUT(out, "\tmul\tr2, r14\n");
  ASM_CODE_OUT(out, "\tsub\tr14, r1\n");
  return 0;
}

static int calc_llshift(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\tshll\tr2, r1\n");
  return 0;
}

static int calc_rashift(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\tshar\tr2, r1\n");
  return 0;
}

static int calc_rlshift(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\tshlr\tr2, r1\n");
  return 0;
}

static int calc_op2(FILE *out, c_type_t type, model_t model, model_t model_arg0, model_t model_arg1)
{
  return -1;
}

static void branch(FILE *out, char *label)
{
  ASM_CODE_OUT(out, "\tbra\t%s\n", label);
}

static int branch_zero(FILE *out, char *label)
{
  ASM_CODE_OUT(out, "\tcmp\t#0, r1\n");
  ASM_CODE_OUT(out, "\tbeq\t%s\n", label);
  return 0;
}

static int branch_nzero(FILE *out, char *label)
{
  ASM_CODE_OUT(out, "\tcmp\t#0, r1\n");
  ASM_CODE_OUT(out, "\tbne\t%s\n", label);
  return 0;
}

static int branch_cmp_eq(FILE *out, char *label, model_t model)
{
  ASM_CODE_OUT(out, "\tcmp\tr2, r1\n");
  ASM_CODE_OUT(out, "\tbeq\t%s\n", label);
  return 0;
}

static int branch_cmp_ne(FILE *out, char *label, model_t model)
{
  ASM_CODE_OUT(out, "\tcmp\tr2, r1\n");
  ASM_CODE_OUT(out, "\tbne\t%s\n", label);
  return 0;
}

static int branch_cmp_lt(FILE *out, char *label, model_t model)
{
  ASM_CODE_OUT(out, "\tcmp\tr2, r1\n");
  ASM_CODE_OUT(out, "\tblt\t%s\n", label);
  return 0;
}

static int branch_cmp_gt(FILE *out, char *label, model_t model)
{
  ASM_CODE_OUT(out, "\tcmp\tr2, r1\n");
  ASM_CODE_OUT(out, "\tbgt\t%s\n", label);
  return 0;
}

static int branch_cmp_le(FILE *out, char *label, model_t model)
{
  ASM_CODE_OUT(out, "\tcmp\tr2, r1\n");
  ASM_CODE_OUT(out, "\tble\t%s\n", label);
  return 0;
}

static int branch_cmp_ge(FILE *out, char *label, model_t model)
{
  ASM_CODE_OUT(out, "\tcmp\tr2, r1\n");
  ASM_CODE_OUT(out, "\tbge\t%s\n", label);
  return 0;
}

static int branch_cmp_ult(FILE *out, char *label, model_t model)
{
  ASM_CODE_OUT(out, "\tcmp\tr2, r1\n");
  ASM_CODE_OUT(out, "\tbnc\t%s\n", label);
  return 0;
}

static int branch_cmp_ugt(FILE *out, char *label, model_t model)
{
  ASM_CODE_OUT(out, "\tcmp\tr2, r1\n");
  ASM_CODE_OUT(out, "\tbgtu\t%s\n", label);
  return 0;
}

static int branch_cmp_ule(FILE *out, char *label, model_t model)
{
  ASM_CODE_OUT(out, "\tcmp\tr2, r1\n");
  ASM_CODE_OUT(out, "\tbleu\t%s\n", label);
  return 0;
}

static int branch_cmp_uge(FILE *out, char *label, model_t model)
{
  ASM_CODE_OUT(out, "\tcmp\tr2, r1\n");
  ASM_CODE_OUT(out, "\tbc\t%s\n", label);
  return 0;
}

static int branch_cmp(FILE *out, char *label, c_type_t type, model_t model)
{
  return -1;
}

static int function_call(FILE *out, char *label)
{
  ASM_CODE_OUT(out, "\tbsr\t%s\n", label);
  return 0;
}

static int function_call_set(FILE *out)
{
  ASM_CODE_OUT(out, "\tmov.l\tr1, r14\n");
  return 0;
}

static int function_call_pointer(FILE *out)
{
  ASM_CODE_OUT(out, "\tjsr\tr14\n");
  return 0;
}

static void function_start(FILE *out)
{
}

static void function_end(FILE *out)
{
  ASM_CODE_OUT(out, "\trts\n");
}

static void function_register_save(FILE *out)
{
  ASM_CODE_OUT(out, "\tpushm\tr7-r13\n");
}

static void function_register_load(FILE *out)
{
  ASM_CODE_OUT(out, "\tpopm\tr7-r13\n");
}

struct asm_code_callback asm_code_callback_rx = {
  NULL, /* type mark */

  4, /* word size */
  4, /* pointer size */
  4, /* funccall args reg number: r1/r2/r3/r4 (See funccall_regs[]) */
  VALUE_FUNCTION_ARGS_NUM, /* funccall args stack number */
  7, /* tmp reg number: r7/r8/r9/r10/r11/r12/r13 (See tmp_regs[]) */
  VALUE_FUNCTION_ARGS_NUM, /* tmp stack number */
  7, /* function register number: r7/r8/r9/r10/r11/r12/r13 (See function_register_save() */
  0, /* function saveparam number: none (See function_start() */
  16, /* stack align size */
  0, /* stack correct size */

  get_value,
  get_value_r1,
  get_address_stack,
  get_address_stack_r1,
  get_address,
  add_address,
  get_r1,
  set_r1,

  memory_load,
  memory_store,
  stack_load,
  stack_store,
  stack_load_r1,
  stack_store_r1,
  stack_expand,
  stack_reduce,

  funccall_reg_load,
  funccall_reg_store,
  tmp_reg_load,
  tmp_reg_save,
  tmp_reg_load_r1,
  tmp_reg_save_r1,

  sign_extension_char,
  sign_extension_uchar,
  sign_extension_short,
  sign_extension_ushort,
  sign_extension_int,
  sign_extension_uint,

  calc_inv,
  calc_minus,

  calc_op1,

  calc_add,
  calc_sub,
  calc_and,
  calc_or,
  calc_xor,

  calc_mul,
  calc_div,
  calc_mod,
  calc_llshift,
  calc_rashift,
  calc_rlshift,

  calc_op2,

  branch,
  branch_zero,
  branch_nzero,

  branch_cmp_eq,
  branch_cmp_ne,

  branch_cmp_lt,
  branch_cmp_gt,
  branch_cmp_le,
  branch_cmp_ge,
  branch_cmp_ult,
  branch_cmp_ugt,
  branch_cmp_ule,
  branch_cmp_uge,

  branch_cmp,

  function_call,
  function_call_set,
  function_call_pointer,

  function_start,
  function_end,
  function_register_save,
  function_register_load,
};
