#include <stdio.h>
#include <stdlib.h>

#ifdef USE_NLLIBC
#include <nllibc.h>
#endif

#include "config.h"
#include "lib.h"
#include "type.h"
#include "asm_symbol.h"
#include "asm_code.h"
#include "asm_code_riscv32.h"

static char *load_by_size(model_t model)
{
  if (model != NULL) {
    switch (model_get_size(model)) {
    case -1: ERREXIT(1);
    case  1: return "lb";
    case  2: return "lh";
    default: break;
    }
  }
  return "lw";
}

static char *store_by_size(model_t model)
{
  if (model != NULL) {
    switch (model_get_size(model)) {
    case -1: ERREXIT(1);
    case  1: return "sb";
    case  2: return "sh";
    default: break;
    }
  }
  return "sw";
}

static void get_value(FILE *out, long value)
{
  if (value & 0x800) {
    ASM_CODE_OUT(out, "\tlui\ta0, 0x%lx\n", ((value >> 20) + 1) & 0xfffff);
    ASM_CODE_OUT(out, "\taddi\ta0, a0, -0x%lx\n", 0x1000 - (value & 0xfff));
  } else {
    ASM_CODE_OUT(out, "\tlui\ta0, 0x%lx\n", (value >> 20) & 0xfffff);
    ASM_CODE_OUT(out, "\taddi\ta0, a0, 0x%lx\n", value & 0xfff);
  }
}

static void get_value_r1(FILE *out, long value)
{
  if (value & 0x800) {
    ASM_CODE_OUT(out, "\tlui\ta1, 0x%lx\n", ((value >> 20) + 1) & 0xfffff);
    ASM_CODE_OUT(out, "\taddi\ta1, a1, -0x%lx\n", 0x1000 - (value & 0xfff));
  } else {
    ASM_CODE_OUT(out, "\tlui\ta1, 0x%lx\n", (value >> 20) & 0xfffff);
    ASM_CODE_OUT(out, "\taddi\ta1, a1, 0x%lx\n", value & 0xfff);
  }
}

static void get_address_stack(FILE *out, int offset)
{
  ASM_CODE_OUT(out, "\taddi\ta0, sp, 0x%x\n", offset);
}

static int get_address_stack_r1(FILE *out, int offset)
{
  ASM_CODE_OUT(out, "\taddi\ta1, sp, 0x%x\n", offset);
  return 0;
}

static void get_address(FILE *out, char *label)
{
  ASM_CODE_OUT(out, "\tlui\ta0, %%hi(%s)\n", label);
  ASM_CODE_OUT(out, "\taddi\ta0, a0, %%lo(%s)\n", label);
}

static void add_address(FILE *out, int offset)
{
  ASM_CODE_OUT(out, "\taddi\ta0, a0, 0x%x\n", offset);
}

static void get_r1(FILE *out)
{
  ASM_CODE_OUT(out, "\tmv\ta0, a1\n");
}

static void set_r1(FILE *out)
{
  ASM_CODE_OUT(out, "\tmv\ta1, a0\n");
}

static void memory_load(FILE *out, int offset, model_t model)
{
  ASM_CODE_OUT(out, "\t%s\ta0, 0x%x(a1)\n", load_by_size(model), offset);
}

static void memory_store(FILE *out, int offset, model_t model)
{
  ASM_CODE_OUT(out, "\t%s\ta0, 0x%x(a1)\n", store_by_size(model), offset);
}

static void stack_load(FILE *out, int offset, model_t model)
{
  ASM_CODE_OUT(out, "\t%s\ta0, 0x%x(sp)\n", load_by_size(model), offset);
}

static void stack_store(FILE *out, int offset, model_t model)
{
  ASM_CODE_OUT(out, "\t%s\ta0, 0x%x(sp)\n", store_by_size(model), offset);
}

static void stack_load_r1(FILE *out, int offset, model_t model)
{
  ASM_CODE_OUT(out, "\t%s\ta1, 0x%x(sp)\n", load_by_size(model), offset);
}

static void stack_store_r1(FILE *out, int offset, model_t model)
{
  ASM_CODE_OUT(out, "\t%s\ta1, 0x%x(sp)\n", store_by_size(model), offset);
}

static void stack_expand(FILE *out, int size)
{
  ASM_CODE_OUT(out, "\taddi\tsp, sp, -0x%x\n", size);
}

static void stack_reduce(FILE *out, int size)
{
  ASM_CODE_OUT(out, "\taddi\tsp, sp, 0x%x\n", size);
}

static char *funccall_regs[] = {
  "a0", "a1", "a2", "a3", "a4", "a5", "a6", "a7"
};

static void funccall_reg_load(FILE *out, int number)
{
  ASM_CODE_OUT(out, "\tmv\ta0, %s\n", funccall_regs[number]);
}

static void funccall_reg_store(FILE *out, int number)
{
  ASM_CODE_OUT(out, "\tmv\t%s, a0\n", funccall_regs[number]);
}

static char *tmp_regs[] = {
  "s0", "s1",
};

static void tmp_reg_load(FILE *out, int number)
{
  ASM_CODE_OUT(out, "\tmv\ta0, %s\n", tmp_regs[number]);
}

static void tmp_reg_save(FILE *out, int number)
{
  ASM_CODE_OUT(out, "\tmv\t%s, a0\n", tmp_regs[number]);
}

static void tmp_reg_load_r1(FILE *out, int number)
{
  ASM_CODE_OUT(out, "\tmv\ta1, %s\n", tmp_regs[number]);
}

static void tmp_reg_save_r1(FILE *out, int number)
{
  ASM_CODE_OUT(out, "\tmv\t%s, a1\n", tmp_regs[number]);
}

static int sign_extension_char(FILE *out)
{
  ASM_CODE_OUT(out, "\tslli\ta0, a0, 24\n");
  ASM_CODE_OUT(out, "\tsrai\ta0, a0, 24\n");
  return 0;
}

static int sign_extension_uchar(FILE *out)
{
  ASM_CODE_OUT(out, "\tslli\ta0, a0, 24\n");
  ASM_CODE_OUT(out, "\tsrli\ta0, a0, 24\n");
  return 0;
}

static int sign_extension_short(FILE *out)
{
  ASM_CODE_OUT(out, "\tslli\ta0, a0, 16\n");
  ASM_CODE_OUT(out, "\tsrai\ta0, a0, 16\n");
  return 0;
}

static int sign_extension_ushort(FILE *out)
{
  ASM_CODE_OUT(out, "\tslli\ta0, a0, 16\n");
  ASM_CODE_OUT(out, "\tsrli\ta0, a0, 16\n");
  return 0;
}

static int sign_extension_int(FILE *out)
{
  return -1;
}

static int sign_extension_uint(FILE *out)
{
  return -1;
}

static int calc_inv(FILE *out, model_t model, model_t model_arg0)
{
  ASM_CODE_OUT(out, "\tseqz\ta0, a0\n");
  return 0;
}

static int calc_minus(FILE *out, model_t model, model_t model_arg0)
{
  ASM_CODE_OUT(out, "\tneg\ta0, a0\n");
  return 0;
}

static int calc_op1(FILE *out, c_type_t type, model_t model, model_t model_arg0)
{
  return -1;
}

static void calc_add(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\tadd\ta0, a0, a1\n");
}

static void calc_sub(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\tsub\ta0, a0, a1\n");
}

static void calc_and(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\tand\ta0, a0, a1\n");
}

static void calc_or(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\tor\ta0, a0, a1\n");
}

static int calc_xor(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\txor\ta0, a0, a1\n");
  return 0;
}

static int calc_mul(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\tmul\ta0, a0, a1\n");
  return 0;
}

static int calc_div(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  if (model_arg0->flags & MODEL_FLAG_UNSIGNED) {
    ASM_CODE_OUT(out, "\tdivu\ta0, a0, a1\n");
  } else {
    ASM_CODE_OUT(out, "\tdiv\ta0, a0, a1\n");
  }
  return 0;
}

static int calc_mod(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  if (model_arg0->flags & MODEL_FLAG_UNSIGNED) {
    ASM_CODE_OUT(out, "\tremu\ta0, a0, a1\n");
  } else {
    ASM_CODE_OUT(out, "\trem\ta0, a0, a1\n");
  }
  return 0;
}

static int calc_llshift(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\tsll\ta0, a0, a1\n");
  return 0;
}

static int calc_rashift(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\tsra\ta0, a0, a1\n");
  return 0;
}

static int calc_rlshift(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\tsrl\ta0, a0, a1\n");
  return 0;
}

static int calc_op2(FILE *out, c_type_t type, model_t model, model_t model_arg0, model_t model_arg1)
{
  return -1;
}

static void branch(FILE *out, char *label)
{
  ASM_CODE_OUT(out, "\tj\t%s\n", label);
}

static int branch_zero(FILE *out, char *label)
{
  ASM_CODE_OUT(out, "\tbeqz\ta0, %s\n", label);
  return 0;
}

static int branch_nzero(FILE *out, char *label)
{
  ASM_CODE_OUT(out, "\tbnez\ta0, %s\n", label);
  return 0;
}

static int branch_cmp_eq(FILE *out, char *label, model_t model)
{
  ASM_CODE_OUT(out, "\tbeq\ta0, a1, %s\n", label);
  return 0;
}

static int branch_cmp_ne(FILE *out, char *label, model_t model)
{
  ASM_CODE_OUT(out, "\tbne\ta0, a1, %s\n", label);
  return 0;
}

static int branch_cmp_lt(FILE *out, char *label, model_t model)
{
  ASM_CODE_OUT(out, "\tblt\ta0, a1, %s\n", label);
  return 0;
}

static int branch_cmp_gt(FILE *out, char *label, model_t model)
{
  ASM_CODE_OUT(out, "\tblt\ta1, a0, %s\n", label);
  return 0;
}

static int branch_cmp_le(FILE *out, char *label, model_t model)
{
  ASM_CODE_OUT(out, "\tble\ta0, a1, %s\n", label);
  return 0;
}

static int branch_cmp_ge(FILE *out, char *label, model_t model)
{
  ASM_CODE_OUT(out, "\tble\ta1, a0, %s\n", label);
  return 0;
}

static int branch_cmp_ult(FILE *out, char *label, model_t model)
{
  ASM_CODE_OUT(out, "\tbltu\ta0, a1, %s\n", label);
  return 0;
}

static int branch_cmp_ugt(FILE *out, char *label, model_t model)
{
  ASM_CODE_OUT(out, "\tbltu\ta1, a0, %s\n", label);
  return 0;
}

static int branch_cmp_ule(FILE *out, char *label, model_t model)
{
  ASM_CODE_OUT(out, "\tbleu\ta0, a1, %s\n", label);
  return 0;
}

static int branch_cmp_uge(FILE *out, char *label, model_t model)
{
  ASM_CODE_OUT(out, "\tbleu\ta1, a0, %s\n", label);
  return 0;
}

static int branch_cmp(FILE *out, char *label, c_type_t type, model_t model)
{
  return -1;
}

static int function_call(FILE *out, char *label)
{
  ASM_CODE_OUT(out, "\tjal\t%s\n", label);
  return 0;
}

static int function_call_set(FILE *out)
{
  ASM_CODE_OUT(out, "\tmv\tra, a0\n");
  return 0;
}

static int function_call_pointer(FILE *out)
{
  ASM_CODE_OUT(out, "\tjalr\tra, 0(ra)\n");
  return 0;
}

static void function_start(FILE *out)
{
  ASM_CODE_OUT(out, "\taddi\tsp, sp, -4\n");
  ASM_CODE_OUT(out, "\tsw\tra, 0(sp)\n");
}

static void function_end(FILE *out)
{
  ASM_CODE_OUT(out, "\tlw\tra, 0(sp)\n");
  ASM_CODE_OUT(out, "\taddi\tsp, sp, 4\n");
  ASM_CODE_OUT(out, "\tret\n\n");
}

static void function_register_save(FILE *out)
{
  ASM_CODE_OUT(out, "\taddi\tsp, sp, -8\n");
  ASM_CODE_OUT(out, "\tsw\ts0, 0(sp)\n");
  ASM_CODE_OUT(out, "\tsw\ts1, 4(sp)\n");
}

static void function_register_load(FILE *out)
{
  ASM_CODE_OUT(out, "\tlw\ts0, 0(sp)\n");
  ASM_CODE_OUT(out, "\tlw\ts1, 4(sp)\n");
  ASM_CODE_OUT(out, "\taddi\tsp, sp, 8\n");
}

struct asm_code_callback asm_code_callback_riscv32 = {
  NULL, /* type mark */

  4, /* word size */
  4, /* pointer size */
  8, /* funccall args reg number: a0/a1/a2/a3/a4/a5/a6/a7 (See funccall_regs[]) */
  VALUE_FUNCTION_ARGS_NUM, /* funccall args stack number */
  2, /* tmp reg number: s0/s1 (See tmp_regs[]) */
  VALUE_FUNCTION_ARGS_NUM, /* tmp stack number */
  2, /* function register number: s0/s1 (See function_register_save() */
  1, /* function saveparam number: ra (See function_start() */
  16, /* stack align size */
  0, /* stack correct size */

  get_value,
  get_value_r1,
  get_address_stack,
  get_address_stack_r1,
  get_address,
  add_address,
  get_r1,
  set_r1,

  memory_load,
  memory_store,
  stack_load,
  stack_store,
  stack_load_r1,
  stack_store_r1,
  stack_expand,
  stack_reduce,

  funccall_reg_load,
  funccall_reg_store,
  tmp_reg_load,
  tmp_reg_save,
  tmp_reg_load_r1,
  tmp_reg_save_r1,

  sign_extension_char,
  sign_extension_uchar,
  sign_extension_short,
  sign_extension_ushort,
  sign_extension_int,
  sign_extension_uint,

  calc_inv,
  calc_minus,

  calc_op1,

  calc_add,
  calc_sub,
  calc_and,
  calc_or,
  calc_xor,

  calc_mul,
  calc_div,
  calc_mod,
  calc_llshift,
  calc_rashift,
  calc_rlshift,

  calc_op2,

  branch,
  branch_zero,
  branch_nzero,

  branch_cmp_eq,
  branch_cmp_ne,

  branch_cmp_lt,
  branch_cmp_gt,
  branch_cmp_le,
  branch_cmp_ge,
  branch_cmp_ult,
  branch_cmp_ugt,
  branch_cmp_ule,
  branch_cmp_uge,

  branch_cmp,

  function_call,
  function_call_set,
  function_call_pointer,

  function_start,
  function_end,
  function_register_save,
  function_register_load,
};
