#include <stdio.h>
#include <stdlib.h>

#ifdef USE_NLLIBC
#include <nllibc.h>
#endif

#include "config.h"
#include "lib.h"
#include "type.h"
#include "asm_symbol.h"
#include "asm_code.h"
#include "asm_code_arm.h"

static char *load_by_size(model_t model)
{
  if (model != NULL) {
    switch (model_get_size(model)) {
    case -1: ERREXIT(1);
    case  1: return "ldrb";
    case  2: return "ldrh";
    default: break;
    }
  }
  return "ldr";
}

static char *store_by_size(model_t model)
{
  if (model != NULL) {
    switch (model_get_size(model)) {
    case -1: ERREXIT(1);
    case  1: return "strb";
    case  2: return "strh";
    default: break;
    }
  }
  return "str";
}

static void _get_value(FILE *out, char *reg, long value)
{
  ASM_CODE_OUT(out, "\tmov\t%s, #0x%lx\n", reg, value & 0xff000000);
  if (value & 0xff0000)
    ASM_CODE_OUT(out, "\tadd\t%s, #0x%lx\n", reg, value & 0xff0000);
  if (value & 0xff00)
    ASM_CODE_OUT(out, "\tadd\t%s, #0x%lx\n", reg, value & 0xff00);
  if (value & 0xff)
    ASM_CODE_OUT(out, "\tadd\t%s, #0x%lx\n", reg, value & 0xff);
}

static void get_value(FILE *out, long value)
{
  _get_value(out, "r0", value);
}

static void get_value_r1(FILE *out, long value)
{
  _get_value(out, "r1", value);
}

static void _add_address(FILE *out, char *reg, int offset)
{
  if (offset & 0xff000000)
    ASM_CODE_OUT(out, "\tadd\t%s, #0x%x\n", reg, offset & 0xff000000);
  if (offset & 0xff0000)
    ASM_CODE_OUT(out, "\tadd\t%s, #0x%x\n", reg, offset & 0xff0000);
  if (offset & 0xff00)
    ASM_CODE_OUT(out, "\tadd\t%s, #0x%x\n", reg, offset & 0xff00);
  if (offset & 0xff)
    ASM_CODE_OUT(out, "\tadd\t%s, #0x%x\n", reg, offset & 0xff);
}

static void get_address_stack(FILE *out, int offset)
{
  ASM_CODE_OUT(out, "\tmov\tr0, sp\n");
  _add_address(out, "r0", offset);
}

static int get_address_stack_r1(FILE *out, int offset)
{
  ASM_CODE_OUT(out, "\tmov\tr1, sp\n");
  _add_address(out, "r1", offset);
  return 0;
}

static void get_address(FILE *out, char *label)
{
  ASM_CODE_OUT(out, "\tldr\tr0, [pc, #0]\n");
  ASM_CODE_OUT(out, "\tmov\tpc, pc\n");
  ASM_CODE_OUT(out, "\t.word\t%s\n", label);
}

static void add_address(FILE *out, int offset)
{
  _add_address(out, "r0", offset);
}

static void get_r1(FILE *out)
{
  ASM_CODE_OUT(out, "\tmov\tr0, r1\n");
}

static void set_r1(FILE *out)
{
  ASM_CODE_OUT(out, "\tmov\tr1, r0\n");
}

static void memory_load(FILE *out, int offset, model_t model)
{
  ASM_CODE_OUT(out, "\t%s\tr0, [r1, #0x%x]\n", load_by_size(model), offset);
}

static void memory_store(FILE *out, int offset, model_t model)
{
  ASM_CODE_OUT(out, "\t%s\tr0, [r1, #0x%x]\n", store_by_size(model), offset);
}

static void stack_load(FILE *out, int offset, model_t model)
{
  ASM_CODE_OUT(out, "\t%s\tr0, [sp, #0x%x]\n", load_by_size(model), offset);
}

static void stack_store(FILE *out, int offset, model_t model)
{
  ASM_CODE_OUT(out, "\t%s\tr0, [sp, #0x%x]\n", store_by_size(model), offset);
}

static void stack_load_r1(FILE *out, int offset, model_t model)
{
  ASM_CODE_OUT(out, "\t%s\tr1, [sp, #0x%x]\n", load_by_size(model), offset);
}

static void stack_store_r1(FILE *out, int offset, model_t model)
{
  ASM_CODE_OUT(out, "\t%s\tr1, [sp, #0x%x]\n", store_by_size(model), offset);
}

static void stack_expand(FILE *out, int size)
{
  ASM_CODE_OUT(out, "\tsub\tsp, sp, #0x%x\n", size);
}

static void stack_reduce(FILE *out, int size)
{
  ASM_CODE_OUT(out, "\tadd\tsp, sp, #0x%x\n", size);
}

static char *funccall_regs[] = {
  "r0", "r1", "r2", "r3"
};

static void funccall_reg_load(FILE *out, int number)
{
  ASM_CODE_OUT(out, "\tmov\tr0, %s\n", funccall_regs[number]);
}

static void funccall_reg_store(FILE *out, int number)
{
  ASM_CODE_OUT(out, "\tmov\t%s, r0\n", funccall_regs[number]);
}

static char *tmp_regs[] = {
  "r4", "r5", "r6",
};

static void tmp_reg_load(FILE *out, int number)
{
  ASM_CODE_OUT(out, "\tmov\tr0, %s\n", tmp_regs[number]);
}

static void tmp_reg_save(FILE *out, int number)
{
  ASM_CODE_OUT(out, "\tmov\t%s, r0\n", tmp_regs[number]);
}

static void tmp_reg_load_r1(FILE *out, int number)
{
  ASM_CODE_OUT(out, "\tmov\tr1, %s\n", tmp_regs[number]);
}

static void tmp_reg_save_r1(FILE *out, int number)
{
  ASM_CODE_OUT(out, "\tmov\t%s, r1\n", tmp_regs[number]);
}

static int sign_extension_char(FILE *out)
{
  ASM_CODE_OUT(out, "\tlsl\tr0, r0, #24\n");
  ASM_CODE_OUT(out, "\tasr\tr0, r0, #24\n");
  return 0;
}

static int sign_extension_uchar(FILE *out)
{
  ASM_CODE_OUT(out, "\tlsl\tr0, r0, #24\n");
  ASM_CODE_OUT(out, "\tlsr\tr0, r0, #24\n");
  return 0;
}

static int sign_extension_short(FILE *out)
{
  ASM_CODE_OUT(out, "\tlsl\tr0, r0, #16\n");
  ASM_CODE_OUT(out, "\tasr\tr0, r0, #16\n");
  return 0;
}

static int sign_extension_ushort(FILE *out)
{
  ASM_CODE_OUT(out, "\tlsl\tr0, r0, #16\n");
  ASM_CODE_OUT(out, "\tlsr\tr0, r0, #16\n");
  return 0;
}

static int sign_extension_int(FILE *out)
{
  return -1;
}

static int sign_extension_uint(FILE *out)
{
  return -1;
}

static int calc_inv(FILE *out, model_t model, model_t model_arg0)
{
  ASM_CODE_OUT(out, "\tmvn\tr0, r0\n");
  return 0;
}

static int calc_minus(FILE *out, model_t model, model_t model_arg0)
{
  ASM_CODE_OUT(out, "\trsb\tr0, r0, #0\n");
  return 0;
}

static int calc_op1(FILE *out, c_type_t type, model_t model, model_t model_arg0)
{
  return -1;
}

static void calc_add(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\tadd\tr0, r0, r1\n");
}

static void calc_sub(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\trsb\tr0, r1, r0\n");
}

static void calc_and(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\tand\tr0, r0, r1\n");
}

static void calc_or(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\torr\tr0, r0, r1\n");
}

static int calc_xor(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\teor\tr0, r0, r1\n");
  return 0;
}

static int calc_mul(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\tmul\tr0, r1, r0\n");
  return 0;
}

static int calc_div(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  return -1; /* use builtin library */
}

static int calc_mod(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  return -1; /* use builtin library */
}

static int calc_llshift(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\tlsl\tr0, r0, r1\n");
  return 0;
}

static int calc_rashift(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\tasr\tr0, r0, r1\n");
  return 0;
}

static int calc_rlshift(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\tlsr\tr0, r0, r1\n");
  return 0;
}

static int calc_op2(FILE *out, c_type_t type, model_t model, model_t model_arg0, model_t model_arg1)
{
  return -1;
}

static void branch(FILE *out, char *label)
{
  ASM_CODE_OUT(out, "\tb\t%s\n", label);
}

static int branch_zero(FILE *out, char *label)
{
  ASM_CODE_OUT(out, "\tcmp\tr0, #0\n");
  ASM_CODE_OUT(out, "\tbeq\t%s\n", label);
  return 0;
}

static int branch_nzero(FILE *out, char *label)
{
  ASM_CODE_OUT(out, "\tcmp\tr0, #0\n");
  ASM_CODE_OUT(out, "\tbne\t%s\n", label);
  return 0;
}

static int branch_cmp_eq(FILE *out, char *label, model_t model)
{
  ASM_CODE_OUT(out, "\tcmp\tr0, r1\n");
  ASM_CODE_OUT(out, "\tbeq\t%s\n", label);
  return 0;
}

static int branch_cmp_ne(FILE *out, char *label, model_t model)
{
  ASM_CODE_OUT(out, "\tcmp\tr0, r1\n");
  ASM_CODE_OUT(out, "\tbne\t%s\n", label);
  return 0;
}

static int branch_cmp_lt(FILE *out, char *label, model_t model)
{
  ASM_CODE_OUT(out, "\tcmp\tr0, r1\n");
  ASM_CODE_OUT(out, "\tblt\t%s\n", label);
  return 0;
}

static int branch_cmp_gt(FILE *out, char *label, model_t model)
{
  ASM_CODE_OUT(out, "\tcmp\tr0, r1\n");
  ASM_CODE_OUT(out, "\tbgt\t%s\n", label);
  return 0;
}

static int branch_cmp_le(FILE *out, char *label, model_t model)
{
  ASM_CODE_OUT(out, "\tcmp\tr0, r1\n");
  ASM_CODE_OUT(out, "\tble\t%s\n", label);
  return 0;
}

static int branch_cmp_ge(FILE *out, char *label, model_t model)
{
  ASM_CODE_OUT(out, "\tcmp\tr0, r1\n");
  ASM_CODE_OUT(out, "\tbge\t%s\n", label);
  return 0;
}

static int branch_cmp_ult(FILE *out, char *label, model_t model)
{
  ASM_CODE_OUT(out, "\tcmp\tr0, r1\n");
  ASM_CODE_OUT(out, "\tbcc\t%s\n", label);
  return 0;
}

static int branch_cmp_ugt(FILE *out, char *label, model_t model)
{
  ASM_CODE_OUT(out, "\tcmp\tr0, r1\n");
  ASM_CODE_OUT(out, "\tbhi\t%s\n", label);
  return 0;
}

static int branch_cmp_ule(FILE *out, char *label, model_t model)
{
  ASM_CODE_OUT(out, "\tcmp\tr0, r1\n");
  ASM_CODE_OUT(out, "\tbls\t%s\n", label);
  return 0;
}

static int branch_cmp_uge(FILE *out, char *label, model_t model)
{
  ASM_CODE_OUT(out, "\tcmp\tr0, r1\n");
  ASM_CODE_OUT(out, "\tbcs\t%s\n", label);
  return 0;
}

static int branch_cmp(FILE *out, char *label, c_type_t type, model_t model)
{
  return -1;
}

static int function_call(FILE *out, char *label)
{
  ASM_CODE_OUT(out, "\tbl\t%s\n", label);
  return 0;
}

static int function_call_set(FILE *out)
{
  ASM_CODE_OUT(out, "\tmov\tr7, r0\n");
  return 0;
}

static int function_call_pointer(FILE *out)
{
  ASM_CODE_OUT(out, "\tmov\tlr, pc\n");
  ASM_CODE_OUT(out, "\tmov\tpc, r7\n");
  return 0;
}

static void function_start(FILE *out)
{
  ASM_CODE_OUT(out, "\tmov\tip, sp\n");
  ASM_CODE_OUT(out, "\tpush\t{fp, ip, lr, pc}\n");
}

static void function_end(FILE *out)
{
  ASM_CODE_OUT(out, "\tldm\tsp, {fp, sp, pc}\n");
}

static void function_register_save(FILE *out)
{
  ASM_CODE_OUT(out, "\tpush\t{r4, r5, r6, r7}\n");
}

static void function_register_load(FILE *out)
{
  ASM_CODE_OUT(out, "\tpop\t{r4, r5, r6, r7}\n");
}

struct asm_code_callback asm_code_callback_arm = {
  "%", /* type mark */

  4, /* word size */
  4, /* pointer size */
  4, /* funccall args reg number: r0/r1/r2/r3 (See funccall_regs[]) */
  VALUE_FUNCTION_ARGS_NUM, /* funccall args stack number */
  3, /* tmp reg number: r4/r5/r6 (See tmp_regs[]) */
  VALUE_FUNCTION_ARGS_NUM, /* tmp stack number */
  4, /* function register number: r4/r5/r6/r7 (See function_register_save() */
  4, /* function saveparam number: fp/ip/lr/pc (See function_start() */
  16, /* stack align size */
  0, /* stack correct size */

  get_value,
  get_value_r1,
  get_address_stack,
  get_address_stack_r1,
  get_address,
  add_address,
  get_r1,
  set_r1,

  memory_load,
  memory_store,
  stack_load,
  stack_store,
  stack_load_r1,
  stack_store_r1,
  stack_expand,
  stack_reduce,

  funccall_reg_load,
  funccall_reg_store,
  tmp_reg_load,
  tmp_reg_save,
  tmp_reg_load_r1,
  tmp_reg_save_r1,

  sign_extension_char,
  sign_extension_uchar,
  sign_extension_short,
  sign_extension_ushort,
  sign_extension_int,
  sign_extension_uint,

  calc_inv,
  calc_minus,

  calc_op1,

  calc_add,
  calc_sub,
  calc_and,
  calc_or,
  calc_xor,

  calc_mul,
  calc_div,
  calc_mod,
  calc_llshift,
  calc_rashift,
  calc_rlshift,

  calc_op2,

  branch,
  branch_zero,
  branch_nzero,

  branch_cmp_eq,
  branch_cmp_ne,

  branch_cmp_lt,
  branch_cmp_gt,
  branch_cmp_le,
  branch_cmp_ge,
  branch_cmp_ult,
  branch_cmp_ugt,
  branch_cmp_ule,
  branch_cmp_uge,

  branch_cmp,

  function_call,
  function_call_set,
  function_call_pointer,

  function_start,
  function_end,
  function_register_save,
  function_register_load,
};
