#include <stdio.h>
#include <stdlib.h>

#ifdef USE_NLLIBC
#include <nllibc.h>
#endif

#include "config.h"
#include "lib.h"
#include "type.h"
#include "asm_symbol.h"
#include "asm_code.h"
#include "asm_code_aarch64.h"

static char *r0_by_size(model_t model)
{
  if (model != NULL) {
    switch (model_get_size(model)) {
    case -1: ERREXIT(1);
    case  1: return "w0";
    case  2: return "w0";
    case  4: return "w0";
    default: break;
    }
  }
  return "x0";
}

static char *r1_by_size(model_t model)
{
  if (model != NULL) {
    switch (model_get_size(model)) {
    case -1: ERREXIT(1);
    case  1: return "w1";
    case  2: return "w1";
    case  4: return "w1";
    default: break;
    }
  }
  return "x1";
}

static char *load_by_size(model_t model)
{
  if (model != NULL) {
    switch (model_get_size(model)) {
    case -1: ERREXIT(1);
    case  1: return "ldrb";
    case  2: return "ldrh";
    case  4: return "ldr";
    default: break;
    }
  }
  return "ldr";
}

static char *store_by_size(model_t model)
{
  if (model != NULL) {
    switch (model_get_size(model)) {
    case -1: ERREXIT(1);
    case  1: return "strb";
    case  2: return "strh";
    case  4: return "str";
    default: break;
    }
  }
  return "str";
}

static void get_value(FILE *out, long value)
{
  ASM_CODE_OUT(out, "\tmov\tx0, #0x%lx\n", value & 0xffff);
  ASM_CODE_OUT(out, "\tmovk\tx0, #0x%lx, lsl #16\n", (value >> 16) & 0xffff);
}

static void get_value_r1(FILE *out, long value)
{
  ASM_CODE_OUT(out, "\tmov\tx1, #0x%lx\n", value & 0xffff);
  ASM_CODE_OUT(out, "\tmovk\tx1, #0x%lx, lsl #16\n", (value >> 16) & 0xffff);
}

static void get_address_stack(FILE *out, int offset)
{
  ASM_CODE_OUT(out, "\tadd\tx0, sp, #0x%x\n", offset);
}

static int get_address_stack_r1(FILE *out, int offset)
{
  ASM_CODE_OUT(out, "\tadd\tx1, sp, #0x%x\n", offset);
  return 0;
}

static void get_address(FILE *out, char *label)
{
  ASM_CODE_OUT(out, "\tadrp\tx0, %s\n", label);
  ASM_CODE_OUT(out, "\tadd\tx0, x0, #:lo12:%s\n", label);
}

static void add_address(FILE *out, int offset)
{
  ASM_CODE_OUT(out, "\tadd\tx0, x0, #0x%x\n", offset);
}

static void get_r1(FILE *out)
{
  ASM_CODE_OUT(out, "\tmov\tx0, x1\n");
}

static void set_r1(FILE *out)
{
  ASM_CODE_OUT(out, "\tmov\tx1, x0\n");
}

static void memory_load(FILE *out, int offset, model_t model)
{
  ASM_CODE_OUT(out, "\t%s\t%s, [x1, #:lo12:0x%x]\n",
	       load_by_size(model), r0_by_size(model), offset);
}

static void memory_store(FILE *out, int offset, model_t model)
{
  ASM_CODE_OUT(out, "\t%s\t%s, [x1, #:lo12:0x%x]\n",
	       store_by_size(model), r0_by_size(model), offset);
}

static void stack_load(FILE *out, int offset, model_t model)
{
  ASM_CODE_OUT(out, "\t%s\t%s, [sp, #:lo12:0x%x]\n",
	       load_by_size(model), r0_by_size(model), offset);
}

static void stack_store(FILE *out, int offset, model_t model)
{
  ASM_CODE_OUT(out, "\t%s\t%s, [sp, #:lo12:0x%x]\n",
	       store_by_size(model), r0_by_size(model), offset);
}

static void stack_load_r1(FILE *out, int offset, model_t model)
{
  ASM_CODE_OUT(out, "\t%s\t%s, [sp, #:lo12:0x%x]\n",
	       load_by_size(model), r1_by_size(model), offset);
}

static void stack_store_r1(FILE *out, int offset, model_t model)
{
  ASM_CODE_OUT(out, "\t%s\t%s, [sp, #:lo12:0x%x]\n",
	       store_by_size(model), r1_by_size(model), offset);
}

static void stack_expand(FILE *out, int size)
{
  ASM_CODE_OUT(out, "\tsub\tsp, sp, #0x%x\n", size);
}

static void stack_reduce(FILE *out, int size)
{
  ASM_CODE_OUT(out, "\tadd\tsp, sp, #0x%x\n", size);
}

static char *funccall_regs[] = {
  "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7"
};

static void funccall_reg_load(FILE *out, int number)
{
  ASM_CODE_OUT(out, "\tmov\tx0, %s\n", funccall_regs[number]);
}

static void funccall_reg_store(FILE *out, int number)
{
  ASM_CODE_OUT(out, "\tmov\t%s, x0\n", funccall_regs[number]);
}

static char *tmp_regs[] = {
  "x16", "x17", "x18", "x19",
};

static void tmp_reg_load(FILE *out, int number)
{
  ASM_CODE_OUT(out, "\tmov\tx0, %s\n", tmp_regs[number]);
}

static void tmp_reg_save(FILE *out, int number)
{
  ASM_CODE_OUT(out, "\tmov\t%s, x0\n", tmp_regs[number]);
}

static void tmp_reg_load_r1(FILE *out, int number)
{
  ASM_CODE_OUT(out, "\tmov\tx1, %s\n", tmp_regs[number]);
}

static void tmp_reg_save_r1(FILE *out, int number)
{
  ASM_CODE_OUT(out, "\tmov\t%s, x1\n", tmp_regs[number]);
}

static int sign_extension_char(FILE *out)
{
  ASM_CODE_OUT(out, "\tsxtb\tx0, w0\n");
  return 0;
}

static int sign_extension_uchar(FILE *out)
{
  ASM_CODE_OUT(out, "\tuxtb\tw0, w0\n");
  return 0;
}

static int sign_extension_short(FILE *out)
{
  ASM_CODE_OUT(out, "\tsxth\tx0, w0\n");
  return 0;
}

static int sign_extension_ushort(FILE *out)
{
  ASM_CODE_OUT(out, "\tuxth\tw0, w0\n");
  return 0;
}

static int sign_extension_int(FILE *out)
{
  ASM_CODE_OUT(out, "\tsxtw\tx0, w0\n");
  return 0;
}

static int sign_extension_uint(FILE *out)
{
  ASM_CODE_OUT(out, "\tmov\tw0, w0\n");
  return 0;
}

static int calc_inv(FILE *out, model_t model, model_t model_arg0)
{
  ASM_CODE_OUT(out, "\tmvn\tx0, x0\n");
  return 0;
}

static int calc_minus(FILE *out, model_t model, model_t model_arg0)
{
  ASM_CODE_OUT(out, "\tneg\tx0, x0\n");
  return 0;
}

static int calc_op1(FILE *out, c_type_t type, model_t model, model_t model_arg0)
{
  return -1;
}

static void calc_add(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\tadd\tx0, x0, x1\n");
}

static void calc_sub(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\tsub\tx0, x0, x1\n");
}

static void calc_and(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\tand\tx0, x0, x1\n");
}

static void calc_or(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\torr\tx0, x0, x1\n");
}

static int calc_xor(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\teor\tx0, x0, x1\n");
  return 0;
}

static int calc_mul(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\tmul\tx0, x0, x1\n");
  return 0;
}

static int calc_div(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\tsdiv\tx0, x0, x1\n");
  return 0;
}

static int calc_mod(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\tsdiv\tx2, x0, x1\n");
  ASM_CODE_OUT(out, "\tmsub\tx0, x2, x1, x0\n");
  return 0;
}

static int calc_llshift(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\tlsl\tx0, x0, x1\n");
  return 0;
}

static int calc_rashift(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\tasr\tx0, x0, x1\n");
  return 0;
}

static int calc_rlshift(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\tlsr\tx0, x0, x1\n");
  return 0;
}

static int calc_op2(FILE *out, c_type_t type, model_t model, model_t model_arg0, model_t model_arg1)
{
  return -1;
}

static void branch(FILE *out, char *label)
{
  ASM_CODE_OUT(out, "\tb\t%s\n", label);
}

static int branch_zero(FILE *out, char *label)
{
  ASM_CODE_OUT(out, "\tcmp\tx0, xzr\n");
  ASM_CODE_OUT(out, "\tb.eq\t%s\n", label);
  return 0;
}

static int branch_nzero(FILE *out, char *label)
{
  ASM_CODE_OUT(out, "\tcmp\tx0, xzr\n");
  ASM_CODE_OUT(out, "\tb.ne\t%s\n", label);
  return 0;
}

static int branch_cmp_eq(FILE *out, char *label, model_t model)
{
  ASM_CODE_OUT(out, "\tcmp\tx0, x1\n");
  ASM_CODE_OUT(out, "\tb.eq\t%s\n", label);
  return 0;
}

static int branch_cmp_ne(FILE *out, char *label, model_t model)
{
  ASM_CODE_OUT(out, "\tcmp\tx0, x1\n");
  ASM_CODE_OUT(out, "\tb.ne\t%s\n", label);
  return 0;
}

static int branch_cmp_lt(FILE *out, char *label, model_t model)
{
  ASM_CODE_OUT(out, "\tcmp\tx0, x1\n");
  ASM_CODE_OUT(out, "\tb.lt\t%s\n", label);
  return 0;
}

static int branch_cmp_gt(FILE *out, char *label, model_t model)
{
  ASM_CODE_OUT(out, "\tcmp\tx0, x1\n");
  ASM_CODE_OUT(out, "\tb.gt\t%s\n", label);
  return 0;
}

static int branch_cmp_le(FILE *out, char *label, model_t model)
{
  ASM_CODE_OUT(out, "\tcmp\tx0, x1\n");
  ASM_CODE_OUT(out, "\tb.le\t%s\n", label);
  return 0;
}

static int branch_cmp_ge(FILE *out, char *label, model_t model)
{
  ASM_CODE_OUT(out, "\tcmp\tx0, x1\n");
  ASM_CODE_OUT(out, "\tb.ge\t%s\n", label);
  return 0;
}

static int branch_cmp_ult(FILE *out, char *label, model_t model)
{
  ASM_CODE_OUT(out, "\tcmp\tx0, x1\n");
  ASM_CODE_OUT(out, "\tb.cc\t%s\n", label);
  return 0;
}

static int branch_cmp_ugt(FILE *out, char *label, model_t model)
{
  ASM_CODE_OUT(out, "\tcmp\tx0, x1\n");
  ASM_CODE_OUT(out, "\tb.hi\t%s\n", label);
  return 0;
}

static int branch_cmp_ule(FILE *out, char *label, model_t model)
{
  ASM_CODE_OUT(out, "\tcmp\tx0, x1\n");
  ASM_CODE_OUT(out, "\tb.ls\t%s\n", label);
  return 0;
}

static int branch_cmp_uge(FILE *out, char *label, model_t model)
{
  ASM_CODE_OUT(out, "\tcmp\tx0, x1\n");
  ASM_CODE_OUT(out, "\tb.cs\t%s\n", label);
  return 0;
}

static int branch_cmp(FILE *out, char *label, c_type_t type, model_t model)
{
  return -1;
}

static int function_call(FILE *out, char *label)
{
  ASM_CODE_OUT(out, "\tbl\t%s\n", label);
  return 0;
}

static int function_call_set(FILE *out)
{
  ASM_CODE_OUT(out, "\tmov\tx15, x0\n");
  return 0;
}

static int function_call_pointer(FILE *out)
{
  ASM_CODE_OUT(out, "\tblr\tx15\n");
  return 0;
}

static void function_start(FILE *out)
{
  ASM_CODE_OUT(out, "\tsub\tsp, sp, #0x10\n");
  ASM_CODE_OUT(out, "\tstr\tx30, [sp, #0x8]\n");
  ASM_CODE_OUT(out, "\tstr\tx29, [sp]\n");
}

static void function_end(FILE *out)
{
  ASM_CODE_OUT(out, "\tldr\tx29, [sp]\n");
  ASM_CODE_OUT(out, "\tldr\tx30, [sp, #0x8]\n");
  ASM_CODE_OUT(out, "\tadd\tsp, sp, #0x10\n");
  ASM_CODE_OUT(out, "\tret\n");
}

static void function_register_save(FILE *out)
{
  ASM_CODE_OUT(out, "\tsub\tsp, sp, #0x20\n");
  ASM_CODE_OUT(out, "\tstr\tx19, [sp, #0x18]\n");
  ASM_CODE_OUT(out, "\tstr\tx18, [sp, #0x10]\n");
  ASM_CODE_OUT(out, "\tstr\tx17, [sp, #0x8]\n");
  ASM_CODE_OUT(out, "\tstr\tx16, [sp]\n");
}

static void function_register_load(FILE *out)
{
  ASM_CODE_OUT(out, "\tldr\tx16, [sp]\n");
  ASM_CODE_OUT(out, "\tldr\tx17, [sp, #0x8]\n");
  ASM_CODE_OUT(out, "\tldr\tx18, [sp, #0x10]\n");
  ASM_CODE_OUT(out, "\tldr\tx19, [sp, #0x18]\n");
  ASM_CODE_OUT(out, "\tadd\tsp, sp, #0x20\n");
}

struct asm_code_callback asm_code_callback_aarch64 = {
  NULL, /* type mark */

  8, /* word size */
  8, /* pointer size */
  8, /* funccall args reg number: x0/x1/x2/x3/x4/x5/x6/x7 (See funccall_regs[]) */
  VALUE_FUNCTION_ARGS_NUM, /* funccall args stack number */
  4, /* tmp reg number: x16/x17/x18/x19 (See tmp_regs[]) */
  VALUE_FUNCTION_ARGS_NUM, /* tmp stack number */
  4, /* function register number: x16/x17/x18/x19 (See function_register_save() */
  2, /* function saveparam number: x30(LR)/x29(FP) (See function_start() */
  16, /* stack align size */
  0, /* stack correct size */

  get_value,
  get_value_r1,
  get_address_stack,
  get_address_stack_r1,
  get_address,
  add_address,
  get_r1,
  set_r1,

  memory_load,
  memory_store,
  stack_load,
  stack_store,
  stack_load_r1,
  stack_store_r1,
  stack_expand,
  stack_reduce,

  funccall_reg_load,
  funccall_reg_store,
  tmp_reg_load,
  tmp_reg_save,
  tmp_reg_load_r1,
  tmp_reg_save_r1,

  sign_extension_char,
  sign_extension_uchar,
  sign_extension_short,
  sign_extension_ushort,
  sign_extension_int,
  sign_extension_uint,

  calc_inv,
  calc_minus,

  calc_op1,

  calc_add,
  calc_sub,
  calc_and,
  calc_or,
  calc_xor,

  calc_mul,
  calc_div,
  calc_mod,
  calc_llshift,
  calc_rashift,
  calc_rlshift,

  calc_op2,

  branch,
  branch_zero,
  branch_nzero,

  branch_cmp_eq,
  branch_cmp_ne,

  branch_cmp_lt,
  branch_cmp_gt,
  branch_cmp_le,
  branch_cmp_ge,
  branch_cmp_ult,
  branch_cmp_ugt,
  branch_cmp_ule,
  branch_cmp_uge,

  branch_cmp,

  function_call,
  function_call_set,
  function_call_pointer,

  function_start,
  function_end,
  function_register_save,
  function_register_load,
};
