#include <stdio.h>
#include <stdlib.h>

#ifdef USE_NLLIBC
#include <nllibc.h>
#endif

#include "config.h"
#include "lib.h"
#include "type.h"
#include "asm_symbol.h"
#include "asm_code.h"
#include "asm_code_osecpu.h"

static int bitsize(model_t model)
{
  if (model != NULL) {
    switch (model_get_size(model)) {
    case -1: ERREXIT(1);
    case  1: return 8;
    case  2: return 16;
    default: break;
    }
  }
  return 32;
}

static void get_value(FILE *out, long value)
{
  ASM_CODE_OUT(out, "\tLIMM(32, R00, 0x%lx);\n", value);
}

static void get_value_r1(FILE *out, long value)
{
  ASM_CODE_OUT(out, "\tLIMM(32, R01, 0x%lx);\n", value);
}

static void get_address_stack(FILE *out, int offset)
{
  ASM_CODE_OUT(out, "\tCP(R00, P00);\n");
  ASM_CODE_OUT(out, "\tLIMM(32, R02, 0x%x);\n", offset);
  ASM_CODE_OUT(out, "\tADD(32, R00, R00, R02);\n");
}

static int get_address_stack_r1(FILE *out, int offset)
{
  ASM_CODE_OUT(out, "\tCP(R01, P00);\n");
  ASM_CODE_OUT(out, "\tLIMM(32, R02, 0x%x);\n", offset);
  ASM_CODE_OUT(out, "\tADD(32, R01, R01, R02);\n");
  return 0;
}

static void get_address(FILE *out, char *label)
{
  ASM_CODE_OUT(out, "\tLIMM(32, R00, %s);\n", label);
}

static void add_address(FILE *out, int offset)
{
  ASM_CODE_OUT(out, "\tLIMM(32, R02, 0x%x);\n", offset);
  ASM_CODE_OUT(out, "\tADD(32, R00, R00, R02);\n");
}

static void get_r1(FILE *out)
{
  ASM_CODE_OUT(out, "\tCP(R00, R01);\n");
}

static void set_r1(FILE *out)
{
  ASM_CODE_OUT(out, "\tCP(R01, R00);\n");
}

static void memory_load(FILE *out, int offset, model_t model)
{
  ASM_CODE_OUT(out, "\tLIMM(32, R02, 0x%x);\n", offset);
  ASM_CODE_OUT(out, "\tADD(32, R02, R01, R02);\n");
  ASM_CODE_OUT(out, "\tCP(P01, R02);\n");
  ASM_CODE_OUT(out, "\tLMEM(%d, R00, 0, P01, 0);\n", bitsize(model));
}

static void memory_store(FILE *out, int offset, model_t model)
{
  ASM_CODE_OUT(out, "\tLIMM(32, R02, 0x%x);\n", offset);
  ASM_CODE_OUT(out, "\tADD(32, R02, R01, R02);\n");
  ASM_CODE_OUT(out, "\tCP(P01, R02);\n");
  ASM_CODE_OUT(out, "\tSMEM(%d, R00, 0, P01, 0);\n", bitsize(model));
}

static void stack_load(FILE *out, int offset, model_t model)
{
  ASM_CODE_OUT(out, "\tCP(R03, P00);\n");
  ASM_CODE_OUT(out, "\tLIMM(32, R02, 0x%x);\n", offset);
  ASM_CODE_OUT(out, "\tADD(32, R02, R03, R02);\n");
  ASM_CODE_OUT(out, "\tCP(P01, R02);\n");
  ASM_CODE_OUT(out, "\tLMEM(%d, R00, 0, P01, 0);\n", bitsize(model));
}

static void stack_store(FILE *out, int offset, model_t model)
{
  ASM_CODE_OUT(out, "\tCP(R03, P00);\n");
  ASM_CODE_OUT(out, "\tLIMM(32, R02, 0x%x);\n", offset);
  ASM_CODE_OUT(out, "\tADD(32, R02, R03, R02);\n");
  ASM_CODE_OUT(out, "\tCP(P01, R02);\n");
  ASM_CODE_OUT(out, "\tSMEM(%d, R00, 0, P01, 0);\n", bitsize(model));
}

static void stack_load_r1(FILE *out, int offset, model_t model)
{
  ASM_CODE_OUT(out, "\tCP(R03, P00);\n");
  ASM_CODE_OUT(out, "\tLIMM(32, R02, 0x%x);\n", offset);
  ASM_CODE_OUT(out, "\tADD(32, R02, R03, R02);\n");
  ASM_CODE_OUT(out, "\tCP(P01, R02);\n");
  ASM_CODE_OUT(out, "\tLMEM(%d, R01, 0, P01, 0);\n", bitsize(model));
}

static void stack_store_r1(FILE *out, int offset, model_t model)
{
  ASM_CODE_OUT(out, "\tCP(R03, P00);\n");
  ASM_CODE_OUT(out, "\tLIMM(32, R02, 0x%x);\n", offset);
  ASM_CODE_OUT(out, "\tADD(32, R02, R03, R02);\n");
  ASM_CODE_OUT(out, "\tCP(P01, R02);\n");
  ASM_CODE_OUT(out, "\tSMEM(%d, R01, 0, P01, 0);\n", bitsize(model));
}

static void stack_expand(FILE *out, int size)
{
  ASM_CODE_OUT(out, "\tCP(R03, P00);\n");
  ASM_CODE_OUT(out, "\tLIMM(32, R02, 0x%x);\n", size);
  ASM_CODE_OUT(out, "\tSUB(32, R03, R03, R02);\n");
  ASM_CODE_OUT(out, "\tCP(P00, R03);\n");
}

static void stack_reduce(FILE *out, int size)
{
  ASM_CODE_OUT(out, "\tCP(R03, P00);\n");
  ASM_CODE_OUT(out, "\tLIMM(32, R02, 0x%x);\n", size);
  ASM_CODE_OUT(out, "\tADD(32, R03, R03, R02);\n");
  ASM_CODE_OUT(out, "\tCP(P00, R03);\n");
}

#if 0
static char *funccall_regs[] = {
  "noreg"
};
#endif

static void funccall_reg_load(FILE *out, int number)
{
}

static void funccall_reg_store(FILE *out, int number)
{
}

#if 0
static char *tmp_regs[] = {
  "noreg",
};
#endif

static void tmp_reg_load(FILE *out, int number)
{
}

static void tmp_reg_save(FILE *out, int number)
{
}

static void tmp_reg_load_r1(FILE *out, int number)
{
}

static void tmp_reg_save_r1(FILE *out, int number)
{
}

static int sign_extension_char(FILE *out)
{
  return -1;
}

static int sign_extension_uchar(FILE *out)
{
  return -1;
}

static int sign_extension_short(FILE *out)
{
  return -1;
}

static int sign_extension_ushort(FILE *out)
{
  return -1;
}

static int sign_extension_int(FILE *out)
{
  return -1;
}

static int sign_extension_uint(FILE *out)
{
  return -1;
}

static int calc_inv(FILE *out, model_t model, model_t model_arg0)
{
  return -1;
}

static int calc_minus(FILE *out, model_t model, model_t model_arg0)
{
  return -1;
}

static int calc_op1(FILE *out, c_type_t type, model_t model, model_t model_arg0)
{
  return -1;
}

static void calc_add(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\tADD(32, R00, R00, R01);\n");
}

static void calc_sub(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\tSUB(32, R00, R00, R01);\n");
}

static void calc_and(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\tAND(32, R00, R00, R01);\n");
}

static void calc_or(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\tOR(32, R00, R00, R01);\n");
}

static int calc_xor(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\tXOR(32, R00, R00, R01);\n");
  return 0;
}

static int calc_mul(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\tMUL(32, R00, R00, R01);\n");
  return 0;
}

static int calc_div(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\tDIV(32, R00, R00, R01);\n");
  return 0;
}

static int calc_mod(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\tMOD(32, R00, R00, R01);\n");
  return 0;
}

static int calc_llshift(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  ASM_CODE_OUT(out, "\tSHL(32, R00, R00, R01);\n");
  return 0;
}

static int calc_rashift(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  return -1;
}

static int calc_rlshift(FILE *out, model_t model, model_t model_arg0, model_t model_arg1)
{
  return -1;
}

static int calc_op2(FILE *out, c_type_t type, model_t model, model_t model_arg0, model_t model_arg1)
{
  return -1;
}

static void branch(FILE *out, char *label)
{
  ASM_CODE_OUT(out, "\tJMP(%s);\n", label);
}

static int branch_zero(FILE *out, char *label)
{
  ASM_CODE_OUT(out, "\tTSTZ(R02, R00);\n");
  ASM_CODE_OUT(out, "\tCND(R02); JMP(%s);\n", label);
  return 0;
}

static int branch_nzero(FILE *out, char *label)
{
  return -1;
}

static int branch_cmp_eq(FILE *out, char *label, model_t model)
{
  ASM_CODE_OUT(out, "\tCMPE(R02, R00, R01);\n");
  ASM_CODE_OUT(out, "\tCND(R02); JMP(%s);\n", label);
  return 0;
}

static int branch_cmp_ne(FILE *out, char *label, model_t model)
{
  return -1;
}

static int branch_cmp_lt(FILE *out, char *label, model_t model)
{
  ASM_CODE_OUT(out, "\tCMPL(R02, R00, R01);\n");
  ASM_CODE_OUT(out, "\tCND(R02); JMP(%s);\n", label);
  return 0;
}

static int branch_cmp_gt(FILE *out, char *label, model_t model)
{
  return -1;
}

static int branch_cmp_le(FILE *out, char *label, model_t model)
{
  return -1;
}

static int branch_cmp_ge(FILE *out, char *label, model_t model)
{
  return -1;
}

static int branch_cmp_ult(FILE *out, char *label, model_t model)
{
  ASM_CODE_OUT(out, "\tCMPL(R02, R00, R01);\n");
  ASM_CODE_OUT(out, "\tCND(R02); JMP(%s);\n", label);
  return 0;
}

static int branch_cmp_ugt(FILE *out, char *label, model_t model)
{
  return -1;
}

static int branch_cmp_ule(FILE *out, char *label, model_t model)
{
  return -1;
}

static int branch_cmp_uge(FILE *out, char *label, model_t model)
{
  return -1;
}

static int branch_cmp(FILE *out, char *label, c_type_t type, model_t model)
{
  return -1;
}

static int function_call(FILE *out, char *label)
{
  ASM_CODE_OUT(out, "\tCALL\t%s\n", label);
  return 0;
}

static int function_call_set(FILE *out)
{
  ASM_CODE_OUT(out, "\tCP(P01, R00);\n");
  return 0;
}

static int function_call_pointer(FILE *out)
{
  ASM_CODE_OUT(out, "\tCALL(P01);\n");
  return 0;
}

static void function_start(FILE *out)
{
  ASM_CODE_OUT(out, "\tENTER\n");
}

static void function_end(FILE *out)
{
  ASM_CODE_OUT(out, "\tLEAVE\n");
  ASM_CODE_OUT(out, "\tRET\n\n");
}

static void function_register_save(FILE *out)
{
}

static void function_register_load(FILE *out)
{
}

struct asm_code_callback asm_code_callback_osecpu = {
  NULL, /* type mark */

  4, /* word size */
  4, /* pointer size */
  0, /* funccall args reg number */
  VALUE_FUNCTION_ARGS_NUM, /* funccall args stack number */
  0, /* tmp reg number: (See tmp_regs[]) */
  VALUE_FUNCTION_ARGS_NUM, /* tmp stack number */
  0, /* function register number */
  0, /* function saveparam number */
  1, /* stack align size */
  0, /* stack correct size */

  get_value,
  get_value_r1,
  get_address_stack,
  get_address_stack_r1,
  get_address,
  add_address,
  get_r1,
  set_r1,

  memory_load,
  memory_store,
  stack_load,
  stack_store,
  stack_load_r1,
  stack_store_r1,
  stack_expand,
  stack_reduce,

  funccall_reg_load,
  funccall_reg_store,
  tmp_reg_load,
  tmp_reg_save,
  tmp_reg_load_r1,
  tmp_reg_save_r1,

  sign_extension_char,
  sign_extension_uchar,
  sign_extension_short,
  sign_extension_ushort,
  sign_extension_int,
  sign_extension_uint,

  calc_inv,
  calc_minus,

  calc_op1,

  calc_add,
  calc_sub,
  calc_and,
  calc_or,
  calc_xor,

  calc_mul,
  calc_div,
  calc_mod,
  calc_llshift,
  calc_rashift,
  calc_rlshift,

  calc_op2,

  branch,
  branch_zero,
  branch_nzero,

  branch_cmp_eq,
  branch_cmp_ne,

  branch_cmp_lt,
  branch_cmp_gt,
  branch_cmp_le,
  branch_cmp_ge,
  branch_cmp_ult,
  branch_cmp_ugt,
  branch_cmp_ule,
  branch_cmp_uge,

  branch_cmp,

  function_call,
  function_call_set,
  function_call_pointer,

  function_start,
  function_end,
  function_register_save,
  function_register_load,
};
